#if NUNIT
using System;
using System.Data;
using System.Data.SqlClient;
using System.Text;

using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using SPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.UnitTests;
namespace gov.va.med.vbecs.DAL.HL7.UnitTests
{
	/// <summary>
	/// Utility methods used across VBECS HL7 unit tests.
	/// </summary>
	public class HL7UnitTestUtil : BaseTest
	{
		// Message Date is now.
		private static readonly string _msgDateTime = System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":","");
		// DateTimeWanted is five days from now.
		private static readonly string _dateTimeWanted = System.DateTime.Now.AddDays(5).ToString("yyyyMMddhhmmsszzz").Replace(":","");
		// Lab Test Result date is today.
		private static readonly string _resultDate = System.DateTime.Now.ToString("yyyyMMdd");

		private static readonly string SELECT_PENDING_ORDER_TEST_STATEMENT = "SELECT TOP 1 OrderedTest.CprsOrderNumber, OrderableTest.CprsOrderableItemId FROM OrderedTest INNER JOIN OrderableTest ON OrderedTest.OrderableTestId = OrderableTest.OrderableTestId WHERE OrderStatusCode = 'P' AND CprsOrderNumber IS NOT NULL AND CprsOrderableItemId IS NOT NULL";
		private static readonly string SELECT_PENDING_ORDER_COMPONENT_STATEMENT = "SELECT TOP 1 OrderedComponent.CprsOrderNumber, ComponentClass.CprsOrderableItemId FROM OrderedComponent INNER JOIN ComponentClass ON OrderedComponent.ComponentClassId = ComponentClass.ComponentClassId WHERE OrderStatusCode = 'P' AND CprsOrderNumber IS NOT NULL AND CprsOrderableItemId IS NOT NULL";
		private static readonly string PLT_COMPONENT_ORDER = "4^PLATELETS^99VBC^^";
		private static readonly string TAS_TEST_ORDER = "1^TAS^99VBC^^";
		private static readonly string KC_DIVISION = "663^VA HEARTLAND - WEST, VISN 15";

		static HL7UnitTestUtil(){}

		/// <summary>
		/// Creates a CPRS new order message for unit testing only.
		/// </summary>
		/// <param name="test">The component or diagnostic test to be ordered.</param>
		/// <param name="divCode">Division Code</param>
		/// <returns>Valid HL7 CPRS order message.</returns>
		public static string GetCprsMessage(string orderControlCode, string orderNum, string groupNum, string test, string divCode, string vistaPatientId)
		{
			return BuildCprsMessage(orderControlCode, orderNum, groupNum, test, divCode, null, vistaPatientId);
		}

		/// <summary>
		/// Creates a Patient Merge message for unit testing only.
		/// </summary>
		/// <param name="fromPatientId">The VistaPatientId from VBECS representing the Merge From patient.</param>
		/// <param name="toPatientId">The VistaPatientId from VBECS representing the Merge To patient.</param>
		/// <returns>Valid HL7 Merge Message.</returns>
		public static string GetPatientMergeMessage(string fromPatientId, string toPatientId)
		{
			return BuildPatientMergeMessage(fromPatientId, toPatientId);
		}

		/// <summary>
		/// Creates a Cprs Cancel Ordered Test message based on an existing pending order in the database for unit testing only. 
		/// </summary>
		/// <returns>Valid HL7 CPRS Cancel order message.</returns>
		public static HL7OmgMessage GetCprsCancelOrderedTestMessage()
		{
			string cprsOrderNumber = string.Empty;
			DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow( SELECT_PENDING_ORDER_TEST_STATEMENT );
			// If nothing found, create a type and screen order.
			if( (dr == null) || (dr.IsNull(Common.VbecsTables.OrderedTest.CprsOrderNumber)) )
			{
				int orderNum = GetRandomNumber(99999);
				int groupNum = orderNum - 1;
				string messageString = GetCprsMessage( "NW", orderNum.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, GetRandomNumber(999999).ToString() );
				HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
				HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, new HL7Interface( InterfaceName.CPRS.ToString() ) ).ProcessNewOrder();
				DataRow drNew = GetPendingOrderFromDatabase( SELECT_PENDING_ORDER_TEST_STATEMENT );
				// If still no pending order in db, return null.
				if( drNew.IsNull(Common.VbecsTables.OrderedTest.CprsOrderNumber))
					return null;
				cprsOrderNumber = drNew["CprsOrderNumber"].ToString();
			}
			else
			{
				cprsOrderNumber = dr["CprsOrderNumber"].ToString();
			}
			DataRow drPendingTest = HL7UnitTestUtil.RunSQLGetDataRow( "EXECUTE spGetOrderedTestbyCprsOrderNumber '" + cprsOrderNumber + "'" );

			return new HL7OmgMessage( BuildCprsMessage( "CA", drPendingTest["CprsOrderNumber"].ToString(), null, drPendingTest["CprsOrderableItemId"].ToString().Trim() + "^UNIT TEST ORDERED TEST^99VBC^^", KC_DIVISION, drPendingTest["VbecsOrderNumber"].ToString(), drPendingTest["VistaPatientId"].ToString()));

		}

		/// <summary>
		/// Creates a Cprs Cancel Ordered Component message based on an existing pending order in the database for unit testing only. 
		/// </summary>
		/// <returns>Valid HL7 CPRS Cancel order message.</returns>
		public static HL7OmgMessage GetCprsCancelOrderedComponentMessage()
		{
			string cprsOrderNumber = string.Empty;
			DataRow dr = GetPendingOrderFromDatabase( SELECT_PENDING_ORDER_COMPONENT_STATEMENT );
			// If nothing found, create a platelet order.

			if( (dr == null) || (dr.IsNull(Common.VbecsTables.OrderedComponent.CprsOrderNumber)) )
			{
				int orderNum = GetRandomNumber(99999);
				int groupNum = orderNum - 1;
				string messageString = GetCprsMessage( "NW", orderNum.ToString(), groupNum.ToString(), PLT_COMPONENT_ORDER, KC_DIVISION, GetRandomNumber(999999).ToString() );
				HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
				HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, new HL7Interface( InterfaceName.CPRS.ToString() ) ).ProcessNewOrder();
				DataRow drNew = GetPendingOrderFromDatabase( SELECT_PENDING_ORDER_COMPONENT_STATEMENT );
				// If still no pending order in db, return null.
				if( drNew.IsNull(Common.VbecsTables.OrderedComponent.CprsOrderNumber))
					return null;
				cprsOrderNumber = drNew["CprsOrderNumber"].ToString();
			}
			else
			{
				cprsOrderNumber = dr["CprsOrderNumber"].ToString();
			}

			DataRow drPendingComponent = HL7UnitTestUtil.RunSQLGetDataRow( "EXECUTE spGetOrderedComponentbyCprsOrderNumber '" + cprsOrderNumber + "'" );

			return new HL7OmgMessage( BuildCprsMessage( "CA", drPendingComponent["CprsOrderNumber"].ToString(), null, drPendingComponent["CprsOrderableItemId"].ToString().Trim() + "^UNIT TEST ORDERED COMPONENT^99VBC^^", KC_DIVISION, drPendingComponent["VbecsOrderNumber"].ToString(), drPendingComponent["VistaPatientId"].ToString()));
		}

		/// <summary>
		/// Creates a previous lab test result message for an existing CPRS Component order in VBECS.
		/// </summary>
		/// <returns></returns>
		public static HL7OmgMessage GetLabPreviousResultsMessage()
		{
			string cprsOrderNumber = string.Empty;
			DataRow dr = GetPendingOrderFromDatabase( SELECT_PENDING_ORDER_COMPONENT_STATEMENT );
			// If nothing found, create a platelet order.
			if( dr.IsNull(Common.VbecsTables.OrderedComponent.CprsOrderNumber))
			{
				int orderNum = GetRandomNumber(99999);
				int groupNum = orderNum - 1;
				string messageString = GetCprsMessage( "NW", orderNum.ToString(), groupNum.ToString(), PLT_COMPONENT_ORDER, KC_DIVISION, GetRandomNumber(999999).ToString() );
				HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
				HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, new HL7Interface( InterfaceName.CPRS.ToString() ) ).ProcessNewOrder();
				DataRow drNew = GetPendingOrderFromDatabase( SELECT_PENDING_ORDER_COMPONENT_STATEMENT );
				// If still no pending order in db, return null.
				if( drNew.IsNull(Common.VbecsTables.OrderedComponent.OrderedComponentGuid))
					return null;
				cprsOrderNumber = drNew["CprsOrderNumber"].ToString();
			}
			else
			{
				cprsOrderNumber = dr["CprsOrderNumber"].ToString();
			}

			DataRow drPendingComponent = HL7UnitTestUtil.RunSQLGetDataRow( "EXECUTE spGetOrderedComponentbyCprsOrderNumber " + cprsOrderNumber );

			DataRow drOrderGroup = HL7UnitTestUtil.RunSQLGetDataRow( "SELECT OrderGroupNumber FROM PatientOrder WHERE PatientOrderGuid = '" + drPendingComponent["PatientOrderGuid"].ToString() + "'" );

			return new HL7OmgMessage( BuildLabPreviousResultsMessage( "PR", drPendingComponent["CprsOrderNumber"].ToString(), drOrderGroup["OrderGroupNumber"].ToString(), drPendingComponent["CprsOrderableItemId"].ToString().Trim() + "^UNIT TEST ORDERED COMPONENT^99VBC^^", KC_DIVISION, drPendingComponent["VbecsOrderNumber"].ToString(), drPendingComponent["VistaPatientId"].ToString()));

		}
		/// <summary>
		/// Static method used throughout the HL7 Unit Test project to generate random numbers.
		/// </summary>
		/// <param name="maxValue">The maximun value for the random number</param>
		/// <returns>A random number up to the maxValue.  Defaults to a maxValue of 999.</returns>
		public static int GetRandomNumber( int maxValue )
		{
			if( maxValue < 0 )
			{
				maxValue = 999;
			}
			return new Random().Next(maxValue);
		}

		private static DataRow GetPendingOrderFromDatabase(string selectStatement)
		{
			DataTable dt = HL7UnitTestUtil.RunSQLGetDataTable( selectStatement );
			if( dt.Rows.Count > 0 )
				return dt.Rows[0];

			return null;
		}

		private static string BuildCprsMessage(string orderControlCode, string orderNum,
			string groupNum, string test, string divCode, string vbecsOrderNum, string vistaPatientId )
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH|^~\\&|OERR|589|VBECS|589|");
			sb.Append(_msgDateTime);
			sb.Append("||OMG^O19|589");
			sb.Append(GetRandomNumber(999999));
			sb.Append("|T|2.4||||AL|US");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID|||^^^^NI~");
			sb.Append(GetRandomNumber(999999999));
			sb.Append("^^^^SS~");
			sb.Append(vistaPatientId);
			sb.Append("^^^^PI||");
			sb.Append("Unittest^Patient");
			sb.Append(GetRandomNumber(999));
			sb.Append("^X||");
			// Patient date of birth
			DateTime datePatientDob = DateTime.Now - System.TimeSpan.FromDays(10000);
			string _patientDob = datePatientDob.ToString("yyyyMMdd").Replace(":","");
			sb.Append( _patientDob + "|M");
			sb.Append(CARRIAGE_RETURN);
			// PV1 segment
			sb.Append("PV1||O|UNIT TEST PATIENT LOCATION|||||||");
			sb.Append(CARRIAGE_RETURN);
			// ORC segment
			sb.Append("ORC|");
			sb.Append(orderControlCode);
			sb.Append("|" + orderNum + "^OR|" + vbecsOrderNum + "^VBEC|" + groupNum + "^OR|||||");
			sb.Append(_dateTimeWanted);
			sb.Append("|53299^TOMLIN^BRIAN||53299^TOMLIN^BRIAN||||");
			if( orderControlCode == "CA" )
			{
				sb.Append("^Order canceled by unit test.|");
			}
			else
			{
				sb.Append("^Reason is unit test.|");
			}
			sb.Append(divCode);
			sb.Append("|||0^NO");
			sb.Append(CARRIAGE_RETURN);
			// OBR segment
			sb.Append("OBR|1|" + orderNum + "^OR||");
			sb.Append(test);
			sb.Append("|||||||L||12231;3041004;2;6479||^^||||||||||||2^^^^");
			sb.Append(_dateTimeWanted);
			sb.Append("^P^IMMEDIATE^Transurethral Resection (TURP)^");
			sb.Append(CARRIAGE_RETURN);
			// NTE segment
			sb.Append("NTE|1||Unit Test Comments");
			sb.Append(CARRIAGE_RETURN);
			// DG1 segment
			sb.Append("DG1|1||^^^^^|||A");
			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		private static string BuildLabPreviousResultsMessage(string orderControlCode, string orderNum,
			string groupNum, string test, string divCode, string vbecsOrderNum, string vistaPatientId )
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH|^~\\&|OERR|589|VBECS|589|");
			sb.Append(_msgDateTime);
			sb.Append("||OMG^O19|589");
			sb.Append(GetRandomNumber(999999));
			sb.Append("|T|2.4||||AL|US");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID|||^^^^NI~");
			sb.Append(GetRandomNumber(999999999));
			sb.Append("^^^^SS~");
			sb.Append(vistaPatientId);
			sb.Append("^^^^PI||");
			sb.Append("Unittest^Patient");
			sb.Append(GetRandomNumber(999));
			sb.Append("^X||19440603|M");
			sb.Append(CARRIAGE_RETURN);
			// PV1 segment
			sb.Append("PV1||O|UNIT TEST PATIENT LOCATION|||||||");
			sb.Append(CARRIAGE_RETURN);
			// ORC segment
			sb.Append("ORC|");
			sb.Append(orderControlCode);
			sb.Append("|" + orderNum + "^OR|" + vbecsOrderNum + "^VBEC|" + groupNum + "^OR|||||");
			sb.Append(_dateTimeWanted);
			sb.Append("|53299^TOMLIN^BRIAN||53299^TOMLIN^BRIAN||||");
			if( orderControlCode == "CA" )
			{
				sb.Append("^Order canceled by unit test.|");
			}
			else
			{
				sb.Append("^Reason is unit test.|");
			}
			sb.Append(divCode);
			sb.Append("|||0^NO");
			sb.Append(CARRIAGE_RETURN);
			// OBR segment
			sb.Append("OBR|1|" + orderNum + "^OR||");
			sb.Append(test);
			sb.Append("|||||||L||12231;3041004;2;6479||^^||||||||||||2^^^^");
			sb.Append(_dateTimeWanted);
			sb.Append("^P^IMMEDIATE^Transurethral Resection (TURP)");
			sb.Append(CARRIAGE_RETURN);
			// OBX segment
			sb.Append("OBX|1||3^HGB^99LRT||13||13.0-18.0||||F|||20001011175388-0600");

			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		private static string BuildPatientMergeMessage(string fromPatientId, string toPatientId)
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH^~|\\&^MPIF TRIGGER^^VBECS ADT^663^");
			sb.Append(_msgDateTime);
			sb.Append("^^ADT~A40^");
			sb.Append(GetRandomNumber(999999));
			sb.Append("^T^2.4^^^AL^AL^");
			sb.Append(CARRIAGE_RETURN);
			// EVN segment
			sb.Append("EVN^^^^^53296~TOMLIN~BRIAN~~~~~~USVHA&&0363~L~~~NI~VA FACILITY ID&663&L^^663");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID^1^V^");
			sb.Append(fromPatientId);
			sb.Append("~~~USVHA&&0363~PI~VA FACILITY ID&663&L^");
			sb.Append(CARRIAGE_RETURN);
			// MGR segment
			sb.Append("MRG^");
			sb.Append(toPatientId);
			sb.Append("~~~USVHA&&0363~PI~VA FACILITY ID&663&L^");
			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		/// <summary>
		/// Creates a DataTable for spInsertPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetInsertPatientSprocTable(System.Guid patientGuid)
		{
			//First, we need to create a patient to merge from and to
			DataTable dt = new DataTable(TABLE.Patient.TableName);

			dt.Columns.Add(TABLE.Patient.PatientGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.Patient.VistaPatientId, typeof(long));
			dt.Columns.Add(TABLE.Patient.PatientSsn, typeof(string));
			dt.Columns.Add(TABLE.Patient.PatientDob, typeof(System.DateTime));
			dt.Columns.Add(TABLE.Patient.PatientDobCode, typeof(string));
			dt.Columns.Add(TABLE.Patient.PatientLastName, typeof(string));
			dt.Columns.Add(TABLE.Patient.PatientFirstName, typeof(string));
			dt.Columns.Add(TABLE.Patient.RecordStatusCode, typeof(string));
			dt.Columns.Add(TABLE.Patient.LastUpdateUser, typeof(string));
			dt.Columns.Add(TABLE.Patient.LastUpdateFunctionId, typeof(int));

			DataRow dr = dt.NewRow();
			dr[TABLE.Patient.PatientGuid] = patientGuid;
			// At the time of unit test coding 5000-6000 were not being used
			dr[TABLE.Patient.VistaPatientId] = GetVistaPatientId();
			dr[TABLE.Patient.PatientSsn] = new Random().Next(0,999999999).ToString("000000000");
			dr[TABLE.Patient.PatientDob] = System.DateTime.Now.AddYears(-55);
			dr[TABLE.Patient.PatientDobCode] = "V";
			dr[TABLE.Patient.PatientLastName] = "Simposon";
			dr[TABLE.Patient.PatientFirstName] = "Homer";
			dr[TABLE.Patient.RecordStatusCode] = "A";
			dr[TABLE.Patient.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dr[TABLE.Patient.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dt.Rows.Add(dr);

			return(dt);
		}

		/// <summary>
		/// Get the next VistaPatientId --
		/// </summary>
		/// <returns></returns>
		public static int GetVistaPatientId()
		{
			DataTable dtIds = HL7UnitTestUtil.RunSQLGetDataTable("SELECT VistaPatientId FROM Patient WHERE VistaPatientId >= 500000 AND VistaPatientId <= 600000 ORDER BY VistaPatientId");

			int vistaId = 0;

			for (int ndx=500000; ndx < 600000; ndx++)
			{
				DataRow[] dtSearch = dtIds.Select("VistaPatientId = " + ndx.ToString());

				if (dtSearch.Length == 0)
				{
					vistaId = ndx;
					break;
				}
			}

			return(vistaId);
		}

		public static DataTable RunSQLGetDataTable( string sqlText )
		{
			using( SqlConnection _cn = new SqlConnection( VbecsConfig.Current.PrimaryDbConnectionString ) )
			{
				SqlCommand _cmd = _cn.CreateCommand();
				_cmd.CommandText = sqlText;
				
				SqlDataAdapter _da = new SqlDataAdapter( _cmd );
				DataTable _dt = new DataTable();

				_cn.Open();
				_da.Fill( _dt );

				return _dt;
			}
		}

		public static DataRow RunSQLGetDataRow( string sqlText )
		{
			DataTable _dt = RunSQLGetDataTable( sqlText );

			if( _dt.Rows.Count != 1 )
			{
				//				Assertion.Fail( String.Format( "{0} rows were fetched in RunSQLGetDataRow instead of 1 row expected.", _dt.Rows.Count.ToString() ) );
				DataTable dt = new DataTable();
				DataRow dr = dt.NewRow();
				return dr;
			}
			return _dt.Rows[0];
		}

		public static void RunSQL( string sqlText )
		{
			using( SqlConnection _cn = new SqlConnection( VbecsConfig.Current.PrimaryDbConnectionString ) )
			{
				SqlCommand _cmd = _cn.CreateCommand();
				_cmd.CommandText = sqlText;

				_cn.Open();
				_cmd.ExecuteNonQuery();
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public static string GetFormattedHL7Message(string message)
		{
			StringBuilder after = new StringBuilder();
			//
			for (int i = 0; i < message.Length; i++)
			{ 
				if ( message[i] == '\n' ) { after.Append(CARRIAGE_RETURN); } 
				else { after.Append(message[i]); }
			}
			//
			return after.ToString();
		}
	}
}
#endif